<?php
require_once __DIR__.'/../db.php';
require_once __DIR__.'/../helpers.php';

// ---------- Filters ----------
$q = trim($_GET['q'] ?? '');                  // search text (name, sku, category, device)
$deviceFilter = isset($_GET['device_id']) && $_GET['device_id'] !== ''
  ? (int)$_GET['device_id'] : null;          // filter by specific device id

// For device dropdown
$allDevices = $pdo->query("SELECT id, name FROM devices ORDER BY name")->fetchAll();

// ---------- Delete (POST only) ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete') {
  verify_csrf();
  $id = (int)($_POST['id'] ?? 0);
  if ($id > 0) {
    $pdo->beginTransaction();
    try {
      $stmt = $pdo->prepare("DELETE FROM products WHERE id=?");
      $stmt->execute([$id]);
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      http_response_code(500);
      echo "Delete failed: " . e($e->getMessage());
      exit;
    }
  }
  // keep current filters after deletion
  $qs = $_SERVER['QUERY_STRING'] ? ('?'.$_SERVER['QUERY_STRING']) : '';
  redirect('products.php'.$qs);
}

// ---------- Build query with filters ----------
// base query + joins (category + device names list)
$sql = "
SELECT p.*, c.name AS category_name,
       GROUP_CONCAT(DISTINCT d2.name ORDER BY d2.name SEPARATOR ', ') AS device_names
FROM products p
LEFT JOIN categories c ON c.id = p.category_id
LEFT JOIN product_devices pd2 ON pd2.product_id = p.id
LEFT JOIN devices d2 ON d2.id = pd2.device_id
";
$where = [];
$params = [];

// filter by a specific device (only products compatible with that device)
if ($deviceFilter) {
  $sql .= " INNER JOIN product_devices pdF ON pdF.product_id = p.id
            INNER JOIN devices dF ON dF.id = pdF.device_id ";
  $where[] = " dF.id = ? ";
  $params[] = $deviceFilter;
}

// text search across name, sku, category, device names
if ($q !== '') {
  // Search against product name/sku, category name, and any device name
  $where[] = "(
      p.name LIKE ?
      OR p.sku LIKE ?
      OR c.name LIKE ?
      OR EXISTS (
          SELECT 1 FROM product_devices pdS
          JOIN devices dS ON dS.id = pdS.device_id
          WHERE pdS.product_id = p.id AND dS.name LIKE ?
      )
  )";
  $like = '%'.$q.'%';
  array_push($params, $like, $like, $like, $like);
}

if ($where) {
  $sql .= " WHERE " . implode(" AND ", $where);
}

$sql .= " GROUP BY p.id
          ORDER BY p.created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll();

include __DIR__.'/header.php';
?>
<div class="card">
  <div class="flex" style="justify-content:space-between; align-items:center;">
    <h2>All Products</h2>
    <a class="btn" href="product_form.php">+ Add Product</a>
  </div>

  <!-- Filters -->
  <form method="get" class="flex" style="gap:12px; align-items:end; margin:12px 0;">
    <div style="min-width:280px;">
      <label class="small">Search</label>
      <input type="text" name="q" placeholder="Search by name, SKU, category, device..."
             value="<?= e($q) ?>">
    </div>
    <div>
      <label class="small">Device</label>
      <select name="device_id">
        <option value="">— Any device —</option>
        <?php foreach ($allDevices as $d): ?>
          <option value="<?= e($d['id']) ?>" <?= $deviceFilter===$d['id'] ? 'selected' : '' ?>>
            <?= e($d['name']) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div>
      <button class="btn" style="margin-top:2px;">Filter</button>
      <?php if ($q !== '' || $deviceFilter): ?>
        <a class="btn secondary" href="products.php" style="margin-top:2px;">Clear</a>
      <?php endif; ?>
    </div>
  </form>

  <table class="table">
    <thead>
      <tr>
        <th>ID</th><th>Name</th><th>SKU</th><th>Category</th><th>Compatible Devices</th><th>Price</th><th>Actions</th>
      </tr>
    </thead>
    <tbody>
    <?php if (!$rows): ?>
      <tr><td colspan="7" class="small">No products found.</td></tr>
    <?php endif; ?>
    <?php foreach ($rows as $r): ?>
      <tr>
        <td><?= e($r['id']) ?></td>
        <td><?= e($r['name']) ?></td>
        <td><?= e($r['sku']) ?></td>
        <td><?= e($r['category_name']) ?></td>
        <td><?= e($r['device_names']) ?></td>
        <td><?= $r['price'] !== null ? e(number_format($r['price'], 2)) : '-' ?></td>
        <td class="actions">
          <a class="btn secondary" href="product_form.php?id=<?= e($r['id']) ?>">Edit</a>
          <form method="post" onsubmit="return confirmDelete('Delete product?')" style="display:inline;">
            <?php csrf_field(); ?>
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="id" value="<?= e($r['id']) ?>">
            <button class="btn danger">Delete</button>
          </form>
        </td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>
</div>

<?php include __DIR__.'/footer.php'; ?>
